/*
 * Decompiled with CFR 0.152.
 */
package com.sqlines.studio.model.filemanager;

import com.sqlines.studio.model.filemanager.FileInfo;
import com.sqlines.studio.model.filemanager.FileManager;
import com.sqlines.studio.model.filemanager.exception.FileProcessException;
import com.sqlines.studio.model.filemanager.exception.InvalidFileException;
import com.sqlines.studio.model.filemanager.listener.FileChangeListener;
import com.sqlines.studio.model.filemanager.listener.RecentFileChangeListener;
import com.sqlines.studio.util.file.FileReader;
import com.sqlines.studio.util.file.FileStatsProvider;
import com.sqlines.studio.util.file.FileWriter;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class FileManagerImpl
implements FileManager {
    private static final Logger log = LogManager.getLogger(FileManagerImpl.class);
    private static final long serialVersionUID = 3941383733647387L;
    private transient FileReader fileReader;
    private transient FileWriter fileWriter;
    private transient FileStatsProvider fileStatsProvider;
    private Set<FileInfo> files = new HashSet<FileInfo>();
    private List<String> recentFilePaths = new LinkedList<String>();
    private transient List<FileChangeListener> filesChangeListeners = new ArrayList<FileChangeListener>(5);
    private transient List<RecentFileChangeListener> recentFilesListeners = new ArrayList<RecentFileChangeListener>(5);

    public FileManagerImpl(FileReader fileReader, FileWriter fileWriter, FileStatsProvider fileStatsProvider) {
        this.fileReader = fileReader;
        this.fileWriter = fileWriter;
        this.fileStatsProvider = fileStatsProvider;
    }

    @Override
    public synchronized void check() {
        for (FileInfo fileInfo : this.files) {
            String path = fileInfo.getPath();
            if (!this.fileStatsProvider.exists(fileInfo.getPath())) {
                log.info("Noticed file has been deleted: " + path);
                this.resetFile(fileInfo);
                continue;
            }
            if (!this.fileWasUpdated(fileInfo)) continue;
            log.info("Noticed file update: " + path);
            this.updateFile(fileInfo);
        }
    }

    private void resetFile(FileInfo fileInfo) {
        String path = fileInfo.getPath();
        this.files.removeIf(file -> file.getPath().equals(path));
        FileChangeListener.Change change = FileChangeListener.Change.builder().changeType(FileChangeListener.Change.ChangeType.FILE_DELETED).filePath(path).build();
        this.filesChangeListeners.forEach(listener -> listener.onChange(change));
    }

    private boolean fileWasUpdated(FileInfo fileInfo) {
        String path = fileInfo.getPath();
        if (!this.fileStatsProvider.exists(path)) {
            return false;
        }
        return this.fileStatsProvider.lastModified(path) != fileInfo.getLastModified();
    }

    private void updateFile(FileInfo fileInfo) {
        long lastModified = this.fileStatsProvider.lastModified(fileInfo.getPath());
        fileInfo.setLastModified(lastModified);
        FileChangeListener.Change change = FileChangeListener.Change.builder().changeType(FileChangeListener.Change.ChangeType.FILE_UPDATED).filePath(fileInfo.getPath()).build();
        this.filesChangeListeners.forEach(listener -> listener.onChange(change));
    }

    @Override
    public synchronized String openFile(String path) {
        try {
            String content = this.fileReader.read(path);
            long lastModified = this.fileStatsProvider.lastModified(path);
            FileInfo fileInfo = new FileInfo(path, lastModified);
            this.files.add(fileInfo);
            log.info("Opened file: " + path);
            this.changeRecentFiles(path);
            return content;
        }
        catch (IOException exc) {
            String errorMsg = "Cannot open file: " + path;
            throw new FileProcessException(errorMsg, exc);
        }
    }

    @Override
    public void add(String path) {
        long lastModified = this.fileStatsProvider.lastModified(path);
        FileInfo file = new FileInfo(path, lastModified);
        this.files.add(file);
        log.info("File added to monitoring: " + path);
    }

    private void changeRecentFiles(String path) {
        if (!this.recentFilePaths.contains(path)) {
            this.recentFilePaths.add(path);
            log.info("Added recent file: " + path);
            this.notifyRecentFileAddListeners(path, this.recentFilePaths.indexOf(path));
        } else {
            int from = this.recentFilePaths.indexOf(path);
            this.recentFilePaths.remove(path);
            this.recentFilePaths.add(0, path);
            this.notifyRecentFileMoveListeners(path, from, 0);
        }
    }

    private void notifyRecentFileAddListeners(String filePath, int index) {
        RecentFileChangeListener.Change change = RecentFileChangeListener.Change.builder().changeType(RecentFileChangeListener.Change.ChangeType.FILE_ADDED).fileIndex(index).filePath(filePath).build();
        this.recentFilesListeners.forEach(listener -> listener.onChange(change));
    }

    private void notifyRecentFileMoveListeners(String filePath, int movedFrom, int movedTo) {
        RecentFileChangeListener.Change change = RecentFileChangeListener.Change.builder().changeType(RecentFileChangeListener.Change.ChangeType.FILE_MOVED).filePath(filePath).movedFrom(movedFrom).movedTo(movedTo).build();
        this.recentFilesListeners.forEach(listener -> listener.onChange(change));
    }

    @Override
    public synchronized void removeFile(String path) {
        this.files.removeIf(file -> file.getPath().equals(path));
    }

    @Override
    public synchronized void removeAllFiles() {
        this.files.clear();
    }

    @Override
    public synchronized void saveFile(String path, String data) {
        FileInfo fileInfo = this.files.stream().filter(file -> file.getPath().equals(path)).findFirst().orElseThrow(() -> new InvalidFileException("No such file: " + path));
        this.saveFile(fileInfo, data);
    }

    private void saveFile(FileInfo fileInfo, String data) {
        String path = "";
        try {
            path = fileInfo.getPath();
            this.fileWriter.write(path, data);
            long lastModified = this.fileStatsProvider.lastModified(path);
            fileInfo.setLastModified(lastModified);
            log.info("File saved: " + path);
        }
        catch (IOException exc) {
            String error = "Cannot write to file: " + path;
            throw new FileProcessException(error, exc);
        }
    }

    @Override
    public synchronized void saveFileAs(String path, String data) {
        this.files.stream().filter(file -> file.getPath().equals(path)).findFirst().ifPresent(file -> {
            throw new InvalidFileException("File already exists: " + path);
        });
        FileInfo fileInfo = new FileInfo(path, this.fileStatsProvider.lastModified(path));
        this.saveFile(fileInfo, data);
        this.add(path);
    }

    @Override
    public int countFiles() {
        return this.files.size();
    }

    @Override
    public synchronized String getRecentFile(int index) {
        return this.recentFilePaths.get(index);
    }

    @Override
    public synchronized void clearRecentFiles() {
        List<String> paths = List.copyOf(this.recentFilePaths);
        this.recentFilePaths.clear();
        paths.forEach(this::notifyRecentFileListenersOfDelete);
        log.info("Recent files cleared");
    }

    private void notifyRecentFileListenersOfDelete(String path) {
        RecentFileChangeListener.Change change = RecentFileChangeListener.Change.builder().changeType(RecentFileChangeListener.Change.ChangeType.FILE_REMOVED).filePath(path).fileIndex(0).build();
        this.recentFilesListeners.forEach(listener -> listener.onChange(change));
    }

    @Override
    public synchronized int countRecentFiles() {
        return this.recentFilePaths.size();
    }

    @Override
    public void addFileChangeListener(FileChangeListener listener) {
        this.filesChangeListeners.add(listener);
    }

    @Override
    public void addRecentFileListener(RecentFileChangeListener listener) {
        this.recentFilesListeners.add(listener);
    }

    private void readObject(ObjectInputStream stream) throws ClassNotFoundException, IOException, NoSuchFieldException, IllegalAccessException {
        this.initFields();
        int filesNumber = stream.readInt();
        for (int i = 0; i < filesNumber; ++i) {
            FileInfo data = (FileInfo)stream.readObject();
            this.files.add(data);
        }
        int recentFilesNumber = stream.readInt();
        for (int i = 0; i < recentFilesNumber; ++i) {
            String data = (String)stream.readObject();
            this.recentFilePaths.add(data);
        }
    }

    private void initFields() {
        this.files = new HashSet<FileInfo>();
        this.recentFilePaths = new LinkedList<String>();
        this.filesChangeListeners = new ArrayList<FileChangeListener>(5);
        this.recentFilesListeners = new ArrayList<RecentFileChangeListener>(5);
    }

    private synchronized void writeObject(ObjectOutputStream stream) throws IOException {
        stream.writeInt(this.files.size());
        for (FileInfo fileInfo : this.files) {
            stream.writeObject(fileInfo);
        }
        stream.writeInt(this.recentFilePaths.size());
        for (String string : this.recentFilePaths) {
            stream.writeObject(string);
        }
    }

    public void setFileReader(FileReader fileReader) {
        this.fileReader = fileReader;
    }

    public void setFileWriter(FileWriter fileWriter) {
        this.fileWriter = fileWriter;
    }

    public void setFileStatsProvider(FileStatsProvider fileStatsProvider) {
        this.fileStatsProvider = fileStatsProvider;
    }

    public void setFiles(Set<FileInfo> files) {
        this.files = files;
    }

    public void setRecentFilePaths(List<String> recentFilePaths) {
        this.recentFilePaths = recentFilePaths;
    }

    public void setFilesChangeListeners(List<FileChangeListener> filesChangeListeners) {
        this.filesChangeListeners = filesChangeListeners;
    }

    public void setRecentFilesListeners(List<RecentFileChangeListener> recentFilesListeners) {
        this.recentFilesListeners = recentFilesListeners;
    }

    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }
        if (!(o instanceof FileManagerImpl)) {
            return false;
        }
        FileManagerImpl other = (FileManagerImpl)o;
        if (!other.canEqual(this)) {
            return false;
        }
        Set<FileInfo> this$files = this.files;
        Set<FileInfo> other$files = other.files;
        if (this$files == null ? other$files != null : !((Object)this$files).equals(other$files)) {
            return false;
        }
        List<String> this$recentFilePaths = this.recentFilePaths;
        List<String> other$recentFilePaths = other.recentFilePaths;
        return !(this$recentFilePaths == null ? other$recentFilePaths != null : !((Object)this$recentFilePaths).equals(other$recentFilePaths));
    }

    protected boolean canEqual(Object other) {
        return other instanceof FileManagerImpl;
    }

    public int hashCode() {
        int PRIME = 59;
        int result = 1;
        Set<FileInfo> $files = this.files;
        result = result * 59 + ($files == null ? 43 : ((Object)$files).hashCode());
        List<String> $recentFilePaths = this.recentFilePaths;
        result = result * 59 + ($recentFilePaths == null ? 43 : ((Object)$recentFilePaths).hashCode());
        return result;
    }
}

